package gov.va.med.mhv.usermgmt.persist.ldap;

import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import netscape.ldap.LDAPAttribute;
import netscape.ldap.LDAPAttributeSet;
import netscape.ldap.LDAPConnection;
import netscape.ldap.LDAPEntry;
import netscape.ldap.LDAPException;

public abstract class LdapEntryModification implements LdapCommand {
	private Map values;
	
	private LdapEntryModification(Map values) {
		this.values = values == null ? Collections.EMPTY_MAP : values;
	}
	
	/**
	 * Factory method for initializing an LdapEntryModification that encapsulates 
	 * the logic for adding an entry.
	 * @param attributeName
	 * @param newValue
	 * @return LdapEntryModification
	 */
	public static LdapEntryModification add(Map values) {
		return new AddModification(values);
	}
	
	/**
	 * Factory method for initializing an LdapEntryModification that encapsulates 
	 * the logic for deleting an entry.
	 * @param dn
	 * @param newValue
	 * @return LdapEntryModification
	 */
	public static LdapEntryModification delete() {
		return new DeleteModification();
	}
	
	/**
	 * Factory method for initializing an LdapEntryModification that encapsulates 
	 * the logic for deleting then adding back an entry.  This is necessary
	 * in those cases where a uid is changed.  Since the uid represents a unique 
	 * identifier, the application would need to have directory administrator privileges.
	 * The alternative is to retain the data, delete the entry, then recreate
	 * with all the old data changed and the new uid created.
	 * @param dn
	 * @param values
	 * @return LdapEntryModification
	 */
	public static LdapEntryModification wipeAndReplace(String dnToWipe, Map values) {
		return new WipeAndReplaceModification(dnToWipe, values);
	}

	protected Map getValues() {
		return values;
	}

	private static class AddModification extends LdapEntryModification {
		private AddModification(Map values) {
			super(values);
		}
		
		public void execute(String distinguishedName, LDAPConnection conn) throws LDAPException {
			LDAPAttributeSet attSet = new LDAPAttributeSet();
			
			for(Iterator i = getValues().entrySet().iterator(); i.hasNext();) {
				Map.Entry entry = (Map.Entry)i.next();
				
				// we assume that the key is a String
				String name = (String)entry.getKey();
				
				// we assume that the value is either a String or a String[]
				Object value = entry.getValue();
				
				LDAPAttribute att = null;				
				if(value instanceof String)
					att = new LDAPAttribute(name, (String)value);
				else if(value instanceof String[])
					att = new LDAPAttribute(name, (String[])value);
				else
					throw new IllegalArgumentException("Unsupported value type: " + value.getClass().getName());
					
				attSet.add(att);
			}
			
			LDAPEntry entry = new LDAPEntry(distinguishedName, attSet);
			conn.add(entry);
		}
	}
	
	private static class DeleteModification extends LdapEntryModification {
		private DeleteModification() {
			super(null);
		}
		
		public void execute(String distinguishedName, LDAPConnection conn) throws LDAPException {
			conn.delete(distinguishedName);
		}
	}
	
	private static class WipeAndReplaceModification extends LdapEntryModification {
		private String dnToWipe;
		
		private WipeAndReplaceModification(String dnToWipe, Map values) {
			super(values);
			this.dnToWipe = dnToWipe;
		}
		
		public void execute(String distinguishedName, LDAPConnection conn) throws LDAPException {
			LDAPEntry current = conn.read(dnToWipe);
			LDAPAttributeSet attSet = current.getAttributeSet();
			
			/* 
			 * Remove the leading identifier portion of the distinguished name.  If left in place,
			 * this would result in multiple values being applied to the same attribute.  For example,
			 * if a user with a distinguished name of uid=joeveteran,ou=people,dc=heva,dc=va,dc=gov
			 * is going to be replaced by a user with a distinguished name of 
			 * uid=janeveteran,ou=people,dc=heva,dc=va,dc=gov, we need to strip off "uid=joeveteran"
			 * to prevent the new entry from having multiple values (stored as a String[]) for the 
			 * uid attribute (i.e., "joeveteran" and "janeveteran").
			 */
			String id = distinguishedName.substring(0, distinguishedName.indexOf('='));
			
			// transfer all of the current attribute name/value pairs to a Map, which can then be altered
			// by the values contained in the Map constructor argument prior to creating a new entry
			Map currentValues = new HashMap();
			for(int i = 0; i < attSet.size(); i++) {
				LDAPAttribute att = attSet.elementAt(i);
				String name = att.getName();
				
				if(name.equalsIgnoreCase(id))
					continue;
				
				String[] values = att.getStringValueArray();
				currentValues.put(name, values);
			}
			
			// all values passed through the constructor supercede those that already exist on the entry
			currentValues.putAll(getValues());
			
			// delete the entry
			new DeleteModification().execute(dnToWipe, conn);
			
			// recreate the entry with the updated values
			new AddModification(currentValues).execute(distinguishedName, conn);
		}
	}
}